#ifndef __TAudioFile__
#define __TAudioFile__

#include <Basics/CCountedObject.hpp>
#include "TWaveFileFormat.hpp"
#include "TAiffFileFormat.hpp"
#include "TAudioBuffer.hpp"
using Exponent::Audio::TAudioBuffer;
using Exponent::Audio::TWaveFileFormat;
using Exponent::Audio::TAiffFileFormat;
using Exponent::Basics::CCountedObject;

namespace Exponent
{
	namespace Audio
	{
		/**
		 * @class TAudioFile TAudioFile.hpp
		 * @brief A representation of an audio file. 
		 *
		 * @date 14/12/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TAudioFile.hpp,v 1.3 2007/02/08 21:08:09 paul Exp $
		 */
		template <class TypeName> class TAudioFile : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 */
			TAudioFile()
			{
				EXPONENT_CLASS_CONSTRUCTION(TAudioFile<TypeName>);
			}

			/**
			 * Destruction
			 */
			virtual ~TAudioFile()
			{
				EXPONENT_CLASS_DESTRUCTION(TAudioFile<TypeName>);
			}

//	===========================================================================

			/**
			 * Get the audio buffer
			 * @retval const TAudioBuffer<TypeName>* The audio buffer
			 */
			const TAudioBuffer<TypeName> *getAudioBuffer() const { return &m_audioBuffer; }

			/**
			 * Get the audio buffer
			 * @retval TAudioBuffer<TypeName>* The audio buffer
			 */
			TAudioBuffer<TypeName> *getMutableAudioBuffer() { return &m_audioBuffer; }

//	===========================================================================

			/**
			 * Read the file
			 * @param filename The name of the file
			 * @retval bool True if loaded properly, false otherwise
			 */
			bool readFile(const CSystemString &filename)
			{
				TWaveFileFormat<TypeName> waveFileFormat;
				if (waveFileFormat.isValidFormat(filename))
				{
					if (!waveFileFormat.openFile(CFileStream::e_input, filename))
					{
						return false;
					}
					
					// Construct the file format
					typename TWaveFileFormat<TypeName>::SAudioFileFormat format;
					if (!waveFileFormat.readHeader(format))
					{
						return false;
					}

					m_audioBuffer.initialise(format.m_numberOfSamples, format.m_numberOfChannels);

					// Now read in the audio data, in range  -1 to +1\n
					if (!waveFileFormat.readData(m_audioBuffer))
					{
						return false;
					}

					return true;
				}
				else
				{
					TAiffFileFormat<TypeName> aiffFileFormat;
					if (!aiffFileFormat.isValidFormat(filename))
					{
						return false;
					}

					if (!aiffFileFormat.openFile(CFileStream::e_input, filename))
					{
						return false;
					}
					
					typename IAudioFileFormat< TypeName >::SAudioFileFormat format;
					if (!aiffFileFormat.readHeader(format))
					{
						return false;
					}

					m_audioBuffer.initialise(format.m_numberOfSamples, format.m_numberOfChannels);

					// Now read in the audio data, in range  -1 to +1\n
					if (!aiffFileFormat.readData(m_audioBuffer))
					{
						return false;
					}

					return true;
				}
				return false;
			}

			/**
			 * Write the file
			 * @param filename The name of the file to write
			 * @param bitDepth 8/16/24
			 * @param sampleRate The sample rate to write at
			 * @retval bool True if written properly, false otherwise
			 * @note This currently only writes wave files
			 */
			bool writeFile(const CSystemString &filename, const unsigned long bitDepth, const unsigned long sampleRate)
			{
				return this->writeFile(filename, bitDepth, sampleRate, m_audioBuffer);
			}

			/**
			 * Write the file
			 * @param filename The name of the file to write
			 * @param bitDepth 8/16/24
			 * @param sampleRate The sample rate to write at
			 * @param buffer The buffer to write
			 * @retval bool True if written properly, false otherwise
			 * @note This currently only writes wave files
			 */
			bool writeFile(const CSystemString &filename, const unsigned long bitDepth, const unsigned long sampleRate, TAudioBuffer<TypeName> &buffer)
			{
				if (!(bitDepth == 8  ||
					  bitDepth == 16 ||
					  bitDepth == 24))
				{
					return false;
				}

				TWaveFileFormat<TypeName> waveFileFormat;

				if (!waveFileFormat.openFile(CFileStream::e_output, filename))
				{
					return false;
				}

				// Now create the format to write to
				typename TWaveFileFormat<TypeName>::SAudioFileFormat format;
				format.m_bitDepth         = bitDepth;   
				format.m_sampleRate	      = sampleRate; 
				format.m_numberOfChannels = buffer.getNumberOfChannels();	
				format.m_numberOfSamples  = buffer.getNumberOfSamples(); 

				if (!waveFileFormat.writeHeader(format))
				{
					return false;
				}

				if (!waveFileFormat.writeData(buffer))
				{
					return false;
				}

				return true;
			}


//	===========================================================================

		protected:

//	===========================================================================
	
			TAudioBuffer<TypeName> m_audioBuffer;				/**< The audio buffer that we write to */
		};

		/** @cond */
		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TAudioFile<TypeName>, TypeName, CCountedObject);
		/** @endcond */
	}
}
#endif	// End of TAudioFile.hpp